//
//  AddressBook.js
//  Widgets
/*
Copyright 2005, Apple Computer, Inc.  All rights reserved.
NOTE:  Use of this source code is subject to the terms of the Software
License Agreement for Mac OS X, which accompanies the code.  Your use
of this source code signifies your agreement to such license terms and
conditions.  Except as expressly granted in the Software License Agreement
for Mac OS X, no other copyright, patent, or other intellectual property
license or right is granted, either expressly or by implication, by Apple.
*/

/*======================================================================
 *	Globals and Global Events
 *======================================================================*/


var gABCurrentCardIndex = 0;
var gSavedSearchString = "";
var gSkipNextCompletion = false;
var gDisplayLog = 0;

var kMinimumHeight = 87;
var kMinimumWidth =  288;
var kMinimumExpandedHeight = 145;
var kDefaultExpandedHeight = 400;

var AddressBookPlugin;

function setup() {
	document.getElementById("pageBottomContent").style.visibility = 'hidden';
	document.getElementById('SearchInputElement').setAttribute("placeholder", getLocalizedString('Name'));
	View.setCurrent(TitleView);
}

if (window.widget) {
	widget.onshow = onShow;
	widget.onhide = onHide;
//	widget.onremove = function () {Preference.removeInstancePreferences();};
}

window.onfocus = function () {
	if (Window.expanded) {
		Scroller.show();
		document.getElementById("pageBottomContent").style.visibility = 'visible';
		Arrows.show();
	} else {
		Scroller.hide();
		document.getElementById("pageBottomContent").style.visibility = 'hidden';
	}
	Window.bottomStyle.background = 'url(Images/Bottom_'+(Window.expanded?'Dark':'Light')+'.png) no-repeat';
	
	// bring back the snapback button
	var element = document.getElementById('SearchInputElementSnapbackIcon');
	if (element != null)
		element.src = 'url("Images/Search_SnapBack.png")';
}

window.onblur = function () {
	var searchElement = document.getElementById('SearchInputElement');
	if ( searchElement.value == "")
		shrink();
	
	Scroller.blur();
	Scroller.hide();
	
	document.getElementById("pageBottomContent").style.visibility = 'hidden';
	Window.bottomStyle.background = 'url(Images/Bottom_Light.png) no-repeat';
	document.addEventListener("mousedown", eatMouse, true);
	Arrows.hide();

	// hid the snapback button
	var element = document.getElementById('SearchInputElementSnapbackIcon');
	if (element != null)
		element.src = 'url("Images/Search_SnapBackCover.png")';
}

function eatMouse(event) {
	event.stopPropagation();
	event.preventDefault();
	document.removeEventListener("mousedown", eatMouse, true);
	document.getElementById('SearchInputElement').focus();
}

/*======================================================================
 *	Window
 *======================================================================*/
// Construct the Address Book widget window	
var Window = function () {
	// Address Book window is created with the shrunk state (defined in Info.plist)
	Window.expanded = false;
	Window.shrunkHeight = window.innerHeight;
	

	Window.page = document.getElementById("pageScroll");
	Window.pageStyle = Window.page.style;
	Window.bottom = document.getElementById("bottombar");
	Window.bottomStyle = Window.bottom.style;

	Window.initialSize = new Position(kMinimumWidth,kMinimumHeight);
	
	var y = Preference.get("windowHeight");
	Window.expandedHeight = y ? y: kDefaultExpandedHeight;

	Window.resizeTo(new Position(kMinimumWidth,Window.initialSize.y),"initialize");
	
}
window.addEventListener('load', Window, false);

Window.currentSize = function () {
	return new Position(kMinimumWidth, window.innerHeight);
}

Window.resizeTo = function ( newSize, status) {
	// limit the size
	if ( newSize.x < kMinimumWidth)	newSize.x = kMinimumWidth;
	if ( ! Window.expanded || (status == "expanding") ) {
		if ( newSize.y < kMinimumHeight)	newSize.y = kMinimumHeight;
	} else {
		if ( newSize.y < kMinimumExpandedHeight)	newSize.y = kMinimumExpandedHeight;
	}

	var currentSize = Window.currentSize();
	if ( ! newSize.equal(currentSize) || status != "tracking" ) {

		if ( Window.expanded) {
			window.resizeTo(newSize.x,newSize.y);
		}

		Window.pageStyle.height = (newSize.y-27) + "px";
		document.getElementById("Middle").style.height = (newSize.y-58) + "px";
		Window.bottomStyle.top = (Window.pageStyle.height - 31) + "px";
		Scroller.refresh();

		if ( ! Window.expanded) {
			window.resizeTo(newSize.x,currentSize.y);
		}

		// set the window size pref at the end
		if ( status != "trackingEnd" && Window.expanded) {
			Window.initialSize.set(newSize);
			Preference.set("windowHeight", newSize.y);
			Window.expandedHeight = newSize.y;
		}
	}
}

Window.hide = function () {
	// noop
}


/*
 *  Events
 */

function onShow()
{
}

function onHide()
{
	AddressBookPlugin.hideLargeType();
}


/*======================================================================
 *	ExpandShrink Animation
 *======================================================================*/

function expand (func) { ExpandShrink.animate(1,func);}
function shrink (func) { ExpandShrink.animate(0,func);}

var ExpandShrink = new Object ();

ExpandShrink.animator = function () {
	if ( ! ExpandShrink._animator) {
		var duration = 400;
		ExpandShrink._animator = new AppleAnimator (duration, 13);
		ExpandShrink._animator.oncomplete = ExpandShrink.onFinish;
	}
	return ExpandShrink._animator;
}

ExpandShrink.adjust = function (animator, now, first, done) {
	Window.resizeTo(new Position(kMinimumWidth, Math.round(now)), "expanding");
}

ExpandShrink.onFinish = function () {
	if (ExpandShrink._animation) {
		delete ExpandShrink._animation;
		ExpandShrink._animation = null;
	}
	if (ExpandShrink._animator) {
		delete ExpandShrink._animator;
		ExpandShrink._animator = null;
	}

	if (Window.expanded) {
		Scroller.show();
		View.focus();
		document.getElementById("pageBottomContent").style.visibility = 'visible';
		Arrows.show();
	} else {
		View.setCurrent(TitleView);
//		Window.bottomStyle.background = 'url(Images/Bottom_Light.png) no-repeat';
	}
	if (ExpandShrink.callback)	setTimeout(ExpandShrink.callback,0);
}

ExpandShrink.animate = function (expand,callback) {
	if (Window.expanded == expand) return;
	Window.expanded = expand;	// todo:is this the best way ?

	ExpandShrink.callback = callback;
	var animator = ExpandShrink.animator();
	animator.oncomplete = ExpandShrink.onFinish;
	
	var from = expand ? kMinimumHeight: Window.currentSize().y;
	var to = expand ? Math.max(Window.expandedHeight, kMinimumExpandedHeight) : kMinimumHeight;
	
	ExpandShrink._animation = new AppleAnimation(from, to, ExpandShrink.adjust);
	animator.addAnimation(ExpandShrink._animation);

	if (expand) {
		View.setCurrent(NameListView);
		Window.bottomStyle.background = 'url(Images/Bottom_Dark.png) no-repeat';
	} else {
		Scroller.hide();
		document.getElementById("pageBottomContent").style.visibility = 'hidden';
		var photoDiv = document.getElementById('PictureDiv');
		photoDiv.src = 'url(Images/PersonSquare.png)';
		Arrows.hide();

		Window.bottomStyle.background = 'url(Images/Bottom_Light.png) no-repeat';
	}

	animator.start();
}

/*======================================================================
 *	Search Field
 *======================================================================*/

var SearchField = function () {
}

SearchField.onSearch = function (searchField) {
	var newSearch = searchField.value;
	var oldSearch = gSavedSearchString;

	var sameSearchAgain = newSearch != oldSearch;

	SearchField.performSearchAndSelectBestMatch(searchField.value, sameSearchAgain);
}

SearchField.performSearchAndSelectBestMatch = function(searchString, selectBestMatch)
{
	gSavedSearchString = searchString;
	
	if (selectBestMatch)
		gABCurrentCardIndex = AddressBookPlugin.searchForStringWithBestMatch(gSavedSearchString);
	else
		gABCurrentCardIndex = AddressBookPlugin.searchForStringWithoutBestMatch(gSavedSearchString);
	
	AddressBookPlugin.displayCardAtIndex(gABCurrentCardIndex);
	View.newSearch();
}


/*======================================================================
 *	Scroller
 *======================================================================*/
var Scroller = function () {
	var element = document.getElementById("scrollbar");
	Scroller.scroller = new AppleVerticalScrollbar(element);
	Scroller.scrollerStyle = element.style;
	Scroller.scrollerStyle.visibility = "hidden";
}
window.addEventListener('load', Scroller, false);

Scroller.focus = function () {
	if (Scroller.view)
		Scroller.view.focus();
}

Scroller.blur = function () {
	if (Scroller.view)
		Scroller.view.blur();
}

Scroller.hide = function () {
	Scroller.scrollerStyle.visibility = "hidden";
}

Scroller.show = function () {
	if (Window.expanded)
		Scroller.scrollerStyle.visibility = "visible";
}

Scroller.setView = function (view) {
	if (Scroller.view) {
		Scroller.view.removeScrollbar(Scroller.scroller);
		delete Scroller.view;
	}
	Scroller.view = new AppleScrollArea(view);
	Scroller.view.addScrollbar(Scroller.scroller);
}

Scroller.refresh = function () {
	if (Scroller.view)
		Scroller.view.refresh();
}

/*======================================================================
 *	Resize Box
 *======================================================================*/

var ResizeBox = new Object ();

ResizeBox.mouseDown = function (event) {
	ResizeBox.clickOffset = Window.currentSize().sub(event);
	document.addEventListener("mousemove", ResizeBox.mouseMove, true); 
	document.addEventListener("mouseup", ResizeBox.mouseUp, true); 
	TrackMouse.tracking = true;
}
ResizeBox.mouseMove = function (event) {
	Window.resizeTo(ResizeBox.clickOffset.add(event),"tracking"); 
}
ResizeBox.mouseUp = function (event) { 
	TrackMouse.tracking = false;
	Window.resizeTo(ResizeBox.clickOffset.add(event), "trackingEnd"); 
    document.removeEventListener("mousemove", ResizeBox.mouseMove, true); 
    document.removeEventListener("mouseup", ResizeBox.mouseUp, true); 
}
ResizeBox.hide = function () {
	var element = document.getElementById("ResizeBox");
	element.style.visibility = "hidden";
}
ResizeBox.show = function () {
	var element = document.getElementById("ResizeBox");
	element.style.visibility = "visible";
}

/*======================================================================
 *	View management
 *======================================================================*/
var View = new Object ();

View.focus = function () {
}

View.setCurrent = function (newView) {
	if ( View._current != newView ) {
		if ( View._current)	View._current.hide();
		newView.activate();
		newView.style.visibility = "visible";
		View._current = newView;
	}
	setTimeout(View.focus,0);
}

View.getCurrent = function () {
	if ( typeof View._current == 'undefined')	View._current = null;
	return View._current;
}

View.isCurrentViewNameListView = function () {
	return View.getCurrent() == NameListView;
}

View.isCurrentViewTitleView = function () {
	return View.getCurrent() == TitleView;
}

View.isCurrentViewDetailView = function () {
	return View.getCurrent() == CardDetailView;
}

View.newSearch = function () {
	View._string = document.getElementById('SearchInputElement').value;

	if ( View._string.length == 0 ) {
		if ( Window.expanded)
			View.clear();
	} else {
		if ( Window.expanded) {
			View._newSearch();
		} else {
			expand(View._newSearch);
		}
	}
}

View._newSearch = function ()
{
	var count = AddressBookPlugin.count();
	var tCount = AddressBookPlugin.totalCount();
	if ( (View.getCurrent().viewKind != NameListView.viewKind) ) { // && (count != tCount) ) {
		
		View._prepareForSearch();
	
		View.setCurrent(NameListView);
	}
	View.getCurrent().refresh();
}

View._prepareForSearch = function () {
	gSkipNextCompletion = false;
	NameListView.resetForNewSearch();
	CardDetailView.resetForNewSearch();
}

View.clear = function () {
	View._prepareForSearch();

	if (Window.expanded)	shrink();
}

/*======================================================================
 *	Name List View
 *======================================================================*/
NameListView = function () {
	var element = document.getElementById('NameListViewBlock');

	NameListView.element = element;
	NameListView.style = element.style;
}
window.addEventListener('load', NameListView, false);

NameListView.resetForNewSearch = function () {
	NameListView.clear();
}

NameListView.viewKind = "NameList";

NameListView.activate = function () {
	if (View.getCurrent() != NameListView.viewKind) {
		NameListView.active = 0;
		window.addEventListener('keypress', NameListView.keyPress, true);
	}
	Arrows.hide();
}

NameListView.hide = function () {
	NameListView.style.visibility = "hidden";
	NameListView.active = 0;
	window.removeEventListener('keypress', NameListView.keyPress, true);
}

NameListView.clear = function () {
	var nameDiv = document.getElementById('NameListSubDiv');
	if (nameDiv) {
		NameListView.element.removeChild(nameDiv);
	}
}

NameListView.refresh = function () {
    var nameDiv;
    var infoDiv;
	var nameList;
    var abCount = 0;
	
    nameList = AddressBookPlugin.nameList();
    abCount = nameList.length;

	// find old name sub div
	nameDiv = document.getElementById('NameListSubDiv');
	if (nameDiv) {
		NameListView.element.removeChild(nameDiv);
	}
	   
	// Setup new div's
	nameDiv = document.createElement('div');
	nameDiv.setAttribute('id', 'NameListSubDiv');
	NameListView.element.appendChild(nameDiv);
		
    if (abCount == 0) {
        loadNoMatchTextDiv(nameDiv);
    } else {
		for (var i=0; i < abCount; i++) {
			gABCurrentCardIndex = NameListView.active = AddressBookPlugin.displayedItemIndex();
			var containerDiv = document.createElement('div');
			var valueDiv = document.createElement('div');
			var leftDiv  = document.createElement('div');
			var rightDiv = document.createElement('div');

			containerDiv.setAttribute('class', 'abNameContainerDiv');
			
			valueDiv.setAttribute('id', 'abNameDiv'+i);
			valueDiv.setAttribute('class', 'abNameDiv');

			leftDiv.setAttribute('class', 'abNameDivLeft');
			leftDiv.setAttribute('id', 'abNameDivLeft'+i);
			rightDiv.setAttribute('class', 'abNameDivRight');
			rightDiv.setAttribute('id', 'abNameDivRight'+i);
			
			if (i == NameListView.active) {
				// Set photo
				var photoDiv = document.getElementById('PictureDiv');
				photoDiv.src = 'url(' + AddressBookPlugin.displayedPhoto() + ')';
				valueDiv.setAttribute('class', 'abNameDiv abNameDivSelected');

				leftDiv.setAttribute('class', 'abNameDivLeft leftHighlight');
				rightDiv.setAttribute('class', 'abNameDivRight rightHighlight');
			}
			valueDiv.innerHTML = nameList[i];
			valueDiv.addEventListener('click', NameListView.mouseUp, false);
			valueDiv.setAttribute('cardIndex', i);
			
			containerDiv.appendChild(leftDiv);
			containerDiv.appendChild(valueDiv);
			containerDiv.appendChild(rightDiv);

			nameDiv.appendChild(containerDiv);
		}
	}
	
	// Update counts
	var cardNumberDiv = document.getElementById('abCardNumberDiv');
	var found = getLocalizedString('@1 found');
	var count = AddressBookPlugin.count();
	
	found = found.replace("@1", "" + count);
	
	cardNumberDiv.innerHTML = found;
	Scroller.setView(nameDiv);
	if (count) {
		var element = document.getElementById('abNameDiv'+NameListView.active);
		if (element == null) {
			element = document.getElementById('abNameDiv' + 0);
		}
		Scroller.view.reveal(element);
	}
	Scroller.refresh();
	
	NameListView.updateNameCompletion();
}

NameListView.keyPress = function(event) {
	var handled = false;
	var nameDiv = document.getElementById('NameListSubDiv');
	var keyCode = event.charCode;
	var oldIndex = NameListView.active;
	var abCount = AddressBookPlugin.count();
	
	gSkipNextCompletion = false;
	
	if (keyCode == 63232 /* up arrow */)
	{
		if (NameListView.active > 0)
			NameListView.active--;
		handled = true;
	}
	else if (keyCode == 63233 /* down arrow */)
	{
		if (NameListView.active < (abCount - 1))
			NameListView.active++;
		handled = true;
	}
	else if (keyCode == 8 || keyCode == 63272) // delete or backspace key
	{
		var searchField = document.getElementById('SearchInputElement');
		
		gSkipNextCompletion = true;
	}

	if (NameListView.active != oldIndex) {
		var nameRecord = document.getElementById('abNameDiv'+NameListView.active);
		
		AddressBookPlugin.displayCardAtIndex(NameListView.active);
		document.getElementById('PictureDiv').src = 'url(' + AddressBookPlugin.displayedPhoto() + ')';
		document.getElementById('abNameDiv'+oldIndex).setAttribute('class', 'abNameDiv');
		document.getElementById('abNameDivLeft'+oldIndex).setAttribute('class', 'abNameDivLeft');
		document.getElementById('abNameDivRight'+oldIndex).setAttribute('class', 'abNameDivRight');
		nameRecord.setAttribute('class', 'abNameDiv abNameDivSelected');

		document.getElementById('abNameDivLeft'+NameListView.active).setAttribute('class', 'abNameDivLeft leftHighlight');
		document.getElementById('abNameDivRight'+NameListView.active).setAttribute('class', 'abNameDivRight rightHighlight');

		Scroller.view.reveal(nameRecord);
		
		NameListView.updateNameCompletionEvenIfEmpty(true);
	}
	if (handled)
	{
		event.stopPropagation();
		event.preventDefault();
	}
}

NameListView.mouseUp = function () {
	gABCurrentCardIndex = event.currentTarget.getAttribute("cardIndex");
	View.setCurrent(CardDetailView);
	CardDetailView.refresh();
}

NameListView.dismiss = function () {
	if (View.getCurrent().viewKind == NameListView.viewKind) {
		gABCurrentCardIndex = NameListView.active;
		View.setCurrent(CardDetailView);
		CardDetailView.refresh();
	}
}

NameListView.updateNameCompletion = function () {
	NameListView.updateNameCompletionEvenIfEmpty(false);
}
NameListView.updateNameCompletionEvenIfEmpty = function (completeEvenIfEmpty) {
	if (!gSkipNextCompletion)
	{
		var searchField = document.getElementById('SearchInputElement');
		var completion = AddressBookPlugin.completionStringForSearchString(gSavedSearchString);
		if (completeEvenIfEmpty || completion != "")
		{
			searchField.value = gSavedSearchString + completion;
			searchField.setSelectionRange(gSavedSearchString.length, searchField.value.length);
		}
	}
	gSkipNextCompletion = false;
}

/*======================================================================
 *	Card Detail View
 *======================================================================*/
CardDetailView = function () {
	var element = document.getElementById('abCardInfoBlock');
	
    CardDetailView.detailBlock = document.getElementById('abCardDetailBlock');
    CardDetailView.element = element;
	CardDetailView.style = element.style;
}
window.addEventListener('load', CardDetailView, false);

CardDetailView.resetForNewSearch = function () {
	CardDetailView.clear();
}

CardDetailView.viewKind = "DetailView";

CardDetailView.activate = function () {
	Arrows.show();
}

CardDetailView.hide = function () {
	var snapbackElement = document.getElementById('SearchInputElementSnapbackIcon');
	if (snapbackElement != null)
	{
		snapbackElement.style.visibility = "hidden";
	}
	Arrows.hide();
	CardDetailView.style.visibility = "hidden";
}

CardDetailView.clear = function () {
    var infoDiv;
    var detailDiv;

    infoDiv = document.getElementById('abCardInfoSubDiv');
    if (infoDiv) {
        CardDetailView.element.removeChild(infoDiv);  
    }
   
    detailDiv = document.getElementById('abCardDetailSubDiv');
    if (detailDiv) {
        CardDetailView.detailBlock.removeChild(detailDiv);
    }
}

CardDetailView.refresh = function ()
{
    var detailDiv;
    var infoDiv;
    var cardNumberDiv

	var snapbackElement = document.getElementById('SearchInputElementSnapbackIcon');
	if (snapbackElement == null)
	{
		snapbackElement = document.createElement('img');

		snapbackElement.setAttribute('id', 'SearchInputElementSnapbackIcon');
		snapbackElement.setAttribute('onmousedown', 'snapbackMouseDown(event, this);');

		snapbackElement.setAttribute('src', 'Images/Search_SnapBack.png');
		
		// append the snapback button to the topContent div
		document.getElementById('topContent').appendChild(snapbackElement);
	}
	snapbackElement.style.visibility = "visible";

    if (AddressBookPlugin.count()) {
		// Tell the model what we want to display

		AddressBookPlugin.displayCardAtIndex(gABCurrentCardIndex);
		
		// update the name/search field
		var searchField = document.getElementById('SearchInputElement');
		searchField.value = AddressBookPlugin.displayedName();
		searchField.setSelectionRange(0, searchField.value.length);
		
		// find old detail sub div
		infoDiv = document.getElementById('abCardInfoSubDiv');
		if (infoDiv) {
			CardDetailView.element.removeChild(infoDiv);  
		}
		
		detailDiv = document.getElementById('abCardDetailSubDiv');
		if (detailDiv) {
			CardDetailView.detailBlock.removeChild(detailDiv);
		}
	   
		// Setup new div's
		infoDiv = document.createElement('div');
		infoDiv.setAttribute('id', 'abCardInfoSubDiv');
		CardDetailView.element.appendChild(infoDiv);
	
		detailDiv = document.createElement('div');
		detailDiv.setAttribute('id', 'abCardDetailSubDiv');
		CardDetailView.detailBlock.appendChild(detailDiv);
	
		if (gABCurrentCardIndex == 999999) {
			cardNumberDiv = document.getElementById('abCardNumberDiv');
			cardNumberDiv.innerHTML = getLocalizedString('No Matching Cards');
			loadNoMatchTextDiv(infoDiv);
		} else {
			// Update counts
			cardNumberDiv = document.getElementById('abCardNumberDiv');
			var beforeIndex = AddressBookPlugin.displayedItemIndex();
			var outOf = getLocalizedString('@1 of @2');
			
			CardDetailView.generateCardDetailDiv(infoDiv, detailDiv);
			gABCurrentCardIndex = AddressBookPlugin.indexOfItem();

			outOf = outOf.replace("@1", (gABCurrentCardIndex + 1));
			outOf = outOf.replace("@2", AddressBookPlugin.totalCount());
			
			cardNumberDiv.innerHTML = outOf;
		}
		Scroller.setView(detailDiv);
		Scroller.refresh();
		Arrows.show();
	}
}

CardDetailView.generateCardDetailDiv = function (infoDiv, detailDiv) {
	var photoDiv;
	var propertyDiv;
	var itemCount;
    
    var currentProperty = null,
        previousProperty = null;
    var propertySeparator = null;
	
	// Set photo
	photoDiv = document.getElementById('PictureDiv');
	photoDiv.src = 'url(' + AddressBookPlugin.displayedPhoto() + ')';

	loadNameBlockDiv(infoDiv);
	
	itemCount = AddressBookPlugin.displayedItemCount();

	for (var i = 0; i < itemCount; i++) {
        currentProperty = AddressBookPlugin.displayedPropertyAtIndex(i);
		if (previousProperty != null && currentProperty != previousProperty && !CardDetailView.bothPropertiesAreInstantMessenger(previousProperty, currentProperty))
        {
            propertySeparator = document.createElement('div');
            propertySeparator.setAttribute('class', 'abPropertySeparatorDiv');
            
            detailDiv.appendChild(propertySeparator);
        }

		propertyDiv = propertyBlockFor(currentProperty,
									   AddressBookPlugin.displayedScriptAtIndex(i),
									   AddressBookPlugin.displayedLabelAtIndex(i),
									   AddressBookPlugin.displayedFullLabelAtIndex(i),
									   AddressBookPlugin.displayedValueAtIndex(i),
									   i);
		detailDiv.appendChild(propertyDiv);
        previousProperty = currentProperty;
	}
}

CardDetailView.bothPropertiesAreInstantMessenger = function (firstProperty, secondProperty) {
	var firstIsIM = CardDetailView.isInstantMessenger(firstProperty);
	var secondIsIM = CardDetailView.isInstantMessenger(secondProperty);
	
	return firstIsIM && secondIsIM;
}

CardDetailView.isInstantMessenger = function (property) {
	return property == "AIMInstant" || property == "JabberInstant" || property == "ICQInstant" || property == "MSNInstant" || property == "YahooInstant";
}

/*======================================================================
 *	Title View
 *======================================================================*/
TitleView = function () {
	var element = document.getElementById('title');

	TitleView.element = element;
	TitleView.style = element.style;
}
window.addEventListener('load', TitleView, false);

TitleView.resetForNewSearch = function () {
}

TitleView.viewKind = "TitleView";

TitleView.activate = function () {
	// noop
}

TitleView.refresh = function () {
	// noop
}

TitleView.hide = function () {
	TitleView.style.visibility = "hidden";
}

/*======================================================================
 *	Arrows
 *======================================================================*/
Arrows = function () {
	Arrows.lArrow = document.getElementById('abLeftButtonDiv');
	Arrows.rArrow = document.getElementById('abRightButtonDiv');
	Arrows.trackingButton = null;
	Arrows.lArrow.addEventListener("mousedown", Arrows.mouseDown, true);
	Arrows.lArrow.addEventListener("mouseover", Arrows.mouseOver, true);
	Arrows.lArrow.addEventListener("mouseout", Arrows.mouseOut, true);
	Arrows.rArrow.addEventListener("mousedown", Arrows.mouseDown, true);
	Arrows.rArrow.addEventListener("mouseover", Arrows.mouseOver, true);
	Arrows.rArrow.addEventListener("mouseout", Arrows.mouseOut, true);
}
window.addEventListener('load', Arrows, false);

Arrows.hide = function () {
	Arrows.lArrow.style.visibility = 'hidden';
	Arrows.rArrow.style.visibility = 'hidden';
}

Arrows.show = function () {
	if (View.getCurrent().viewKind == CardDetailView.viewKind) {
		var parentVis = document.getElementById("pageBottomContent").style.visibility;
		Arrows.lArrow.style.visibility = parentVis;
		Arrows.rArrow.style.visibility = parentVis;
	}
}

Arrows.trackMouse = function (inEvent) {
	// we use this event listener for the global document instead of an event 
	// handler on the button <IMG> so that we can know to stop the tracking
	// process even if the mouseup happens outside of the button
	
	if (null != Arrows.trackingButton)
		Arrows.trackingButton.src = "Images/"+Arrows.trackingButton.tag+".png";
	
	// note that inEvent.target will point to the IMG object only if this is a 
	// mouseup inside the button, and also make sure we don't respond to mouseup
	// in another button, hence the extra check for this event being inside the 
	// button being tracked
	
	if (inEvent.target == Arrows.trackingButton)
	{
		if (inEvent.target.tag == "Previous")
		{
			Arrows.clickedLeftButton(event);
		}
		else if (inEvent.target.tag == "Next")
		{
			Arrows.clickedRightButton(event);;
		}
	}
	
	inEvent.stopPropagation();
	inEvent.preventDefault();
	
	document.removeEventListener ("mouseup", Arrows.trackMouse, true);
	Arrows.trackingButton = null;
}

Arrows.mouseDown = function (inEvent) {
	var element = inEvent.currentTarget;
	inEvent.target.src = "Images/"+element.tag+" Pressed.png";
	inEvent.stopPropagation();
	inEvent.preventDefault();
	document.addEventListener ("mouseup", Arrows.trackMouse, true);
	Arrows.trackingButton = element;
}

Arrows.mouseOut = function (inEvent) {
	var element = inEvent.currentTarget;
	inEvent.target.src = "Images/"+element.tag+".png";
}

Arrows.mouseOver = function (inEvent) {
	var element = inEvent.currentTarget;
	if (inEvent.currentTarget == Arrows.trackingButton)
		inEvent.target.src = "Images/"+element.tag+" Pressed.png";
}

Arrows.clickedLeftButton = function (event) {
    if (gABCurrentCardIndex == 999999) {
        return;
    }
    gABCurrentCardIndex--;

	View.setCurrent(CardDetailView);
    if (gABCurrentCardIndex < 0) {
        var max = AddressBookPlugin.totalCount();
        gABCurrentCardIndex = max - 1;
    }
        
    CardDetailView.refresh();
}

Arrows.clickedRightButton = function (event) {
    var max = AddressBookPlugin.totalCount();
    
    if (gABCurrentCardIndex == 999999) {
        return;
    }
    
	View.setCurrent(CardDetailView);
    gABCurrentCardIndex++;

    if (gABCurrentCardIndex >= max)
        gABCurrentCardIndex = 0;
        
    CardDetailView.refresh();
}

/*======================================================================
 *	Preference & Localization Utilities
 *======================================================================*/
var Preference = new Object();
Preference.instancePrefs = new Array();

Preference.get = function (key) {
	if (window.widget) {
		var str = widget.preferenceForKey(_createkey(key));
		if ( str == null) {
			str = widget.preferenceForKey(key);
			if ( str == null) {
				str = localizedStrings[key];
			}
		}
		return str;
	}
}

Preference.set = function (key,value) {
	if (window.widget) {
		var instanceKey = _createkey(key);
		widget.setPreferenceForKey( value,  instanceKey);
		widget.setPreferenceForKey( value,  key);
		Preference.instancePrefs[instanceKey] = true;
	}
}

Preference.removeInstancePreferences = function () {
	if (window.widget) {
		for ( key in Preference.instancePrefs ) {
			widget.setPreferenceForKey(null, key);
		}
	}
}

function _createkey(key) {
        return widget.identifier + "-" + key;
}

function getLocalizedString (key)
{
	try {
		return localizedStrings[key];
	} catch (ex) {}

	return key;
}

/*======================================================================
 *	Tracking Mouse
 *======================================================================*/
TrackMouse = function (view) {
	if ( ! TrackMouse.tracking) {
		TrackMouse.tracking = true;
		view.highlight(true);
		TrackMouse.element = view.element;
		TrackMouse.view    = view;
		TrackMouse.mouseWithin = true;

		TrackMouse.element.addEventListener("mouseover", TrackMouse.mouseOver, true);
		TrackMouse.element.addEventListener("mouseout",  TrackMouse.mouseOut,  true);
		document.addEventListener("mouseup",  TrackMouse.mouseUp,   true);
	}
}

TrackMouse.mouseOver = function (event) {
	TrackMouse.mouseWithin = true;
	TrackMouse.view.highlight(true);
}

TrackMouse.mouseOut = function (event) {
	TrackMouse.mouseWithin = false;
	TrackMouse.view.highlight(false);
}

TrackMouse.mouseUp = function (event) {
	TrackMouse.tracking = false;
	TrackMouse.element.removeEventListener("mouseover", TrackMouse.mouseOver, true);
	TrackMouse.element.removeEventListener("mouseout",  TrackMouse.mouseOut,  true);
	document.removeEventListener("mouseup",  TrackMouse.mouseUp,   true);
	TrackMouse.view.highlight(false);
	if ( TrackMouse.mouseWithin)
		TrackMouse.view.clicked();
}

/*======================================================================
 *	UTILITIES
 *======================================================================*/
function max(a,b) { return (a>b) ? a : b; }
function min(a,b) { return (a<b) ? a : b; }
function abs(a)   { return (a>0) ? a : -a; }

/*======================================================================
 *	Position object
 *======================================================================*/
var Position = function (x,y) { this.x = x; this.y = y;}
Position.prototype = {
set : function (pos) { this.x = pos.x; this.y = pos.y;},
add : function (pos) { return new Position(this.x+pos.x, this.y+pos.y);},
sub : function (pos) { return new Position(this.x-pos.x, this.y-pos.y);},
equal : function (pos) { return (this.x==pos.x) && (this.y==pos.y); },
toString : function () { return "["+this.x+","+this.y+"]";}
}

/*======================================================================
 *  Debug
 *======================================================================*/

function debug(msg) {
     if (!debug.box) {
          debug.box = document.createElement('div');
          debug.box.setAttribute('style', 'background-color: white; ' +
                                                  'font-family: monospace; ' +
                                                  'border: solid black 3px; ' +
                                                  'padding: 10px;');
          document.body.appendChild(debug.box);
          debug.box.innerHTML = '<h1 style="text-align:center">Debug Output</h1>';
     }
     
     var p = document.createElement('p');
     p.appendChild(document.createTextNode(msg));
     debug.box.appendChild(p);
}

/*======================================================================
 *  UI Building
 *======================================================================*/

// Loads the interactive bits of the card:
// Returns a div containing the label and value
function propertyBlockFor (property, script, label, fullLabel, value, index)
{
	var enclosingDiv;
	var labelDiv;
	var valueDiv;
	
	enclosingDiv = document.createElement('div');
	enclosingDiv.setAttribute('class', 'abPropertyDiv');
	
	labelDiv = document.createElement('div');
	labelDiv.setAttribute('class', 'abLabelDiv');
	labelDiv.setAttribute('id', 'label-' + index);
	
	var html = label;
	if (script)
		html = "<A HREF=\"#\" onMouseOut=\"unhighlight(" + index + ");\" onMouseOver=\"highlight(" + index + ");\" onClick=\"" + script + "\">" + label + "</A>";
	labelDiv.innerHTML = html;
	
	valueDiv = document.createElement('div');
	valueDiv.setAttribute('class', 'abValueDiv');
	valueDiv.setAttribute('id', 'value-' + index);
	
	html = value;
	if (script)
		html = "<A HREF=\"#\" onMouseOut=\"unhighlight(" + index + ");\" onMouseOver=\"highlight(" + index + ");\" onClick=\"" + script + "\">" + value + "</A>";
    
    // add the service to the instant messenger properties
    if (property == "AIMInstant" || (property == "JabberInstant") || (property == "MSNInstant") || (property == "YahooInstant") || (property == "ICQInstant"))
        html = getLocalizedString(property).replace("@1", html);
    
	valueDiv.innerHTML = html;
	
	enclosingDiv.appendChild(labelDiv);
	enclosingDiv.appendChild(valueDiv);
	return enclosingDiv;     
}

function updateTextDecorationAtIndex(index, decoration)
{
	var labelElement = document.getElementById("label-" + index);
	labelElement.style.textDecoration = decoration;
	var valueElement = document.getElementById("value-" + index);
	valueElement.style.textDecoration = decoration;
}

function highlight(index)
{
	updateTextDecorationAtIndex(index, "underline");
}

function unhighlight(index)
{
	updateTextDecorationAtIndex(index, "none");
}

function truncateElementHTML(element, maxHeight)
{
	if (element.offsetHeight <= maxHeight)
		return;
	
    var string = element.firstChild.innerHTML;
    var len;
    for (len = string.length; len > 0; len--)
    {
        string = string.substring(0, len);
        element.firstChild.innerHTML = string + "&hellip;";
		if (element.offsetHeight <= maxHeight)
			return;
    }
}

function truncateElementText(element, maxHeight)
{
	if (element.offsetHeight <= maxHeight)
		return;
	
    var string = element.innerText;
    var len;
    for (len = string.length; len > 0; len--)
    {
        string = string.substring(0, len);
        element.innerHTML = string + "&hellip;";
		if (element.offsetHeight <= maxHeight)
			return;
    }
}

// Adds the Name and Org fields to the card
function loadNameBlockDiv (textDiv)
{
	var tr;
	var altNameDiv;
	
	altNameDiv = document.createElement('div');
	altNameDiv.setAttribute('id', 'abPersonAltName');
	
	textDiv.appendChild(altNameDiv);
	
	var altName = AddressBookPlugin.displayedAltName();
	if (altName == null)
		altName = "";
	
	altNameDiv.innerText = altName;
}

function loadNoMatchTextDiv (textDiv) 
{
	var noMatchDiv = document.createElement('div');
	noMatchDiv.setAttribute('id', 'abNoMatchDiv');
	noMatchDiv.innerText = getLocalizedString('No Matching Cards');
	textDiv.appendChild(noMatchDiv);
	var photoDiv = document.getElementById('PictureDiv');
	photoDiv.src = 'url(Images/PersonSquare.png)';
}

/*======================================================================
 *  Event Handlers
 *======================================================================*/

var snapbackIconMouseOver = false;
var snapbackIconMouseDown = false;

function snapbackMouseDown (event, img)
{
	snapbackIconMouseDown = true;
	
	var element = document.getElementById('SearchInputElementSnapbackIcon');
	element.src = 'url("Images/Search_SnapBackPressed.png")';

	document.addEventListener("mouseup", snapbackMouseUp, true);
	document.addEventListener("mousemove", snapbackMouseMove, true);
	element.addEventListener ("mouseover", snapbackMouseOver, true);
	element.addEventListener ("mouseout", snapbackMouseOut, true);
	
	element.setAttribute ("mouseInButton", true);
	
	event.stopPropagation();
	event.preventDefault();
}

function snapbackMouseUp (event)
{
	var element = document.getElementById('SearchInputElementSnapbackIcon');

	document.removeEventListener ("mouseup", snapbackMouseUp, true);
	document.removeEventListener ("mousemove", snapbackMouseMove, true);
	element.removeEventListener ("mouseover", snapbackMouseOver, true);
	element.removeEventListener ("mouseout", snapbackMouseOut, true);

	if (element.mouseInButton)
	{
		element.src = 'url("Images/Search_SnapBack.png")';

		var searchField = document.getElementById('SearchInputElement');
		searchField.focus();
		searchField.value = gSavedSearchString;
		// make sure the carret is at the end
		searchField.setSelectionRange(gSavedSearchString.length, gSavedSearchString.length);
		SearchField.performSearchAndSelectBestMatch(gSavedSearchString, false);
		View.newSearch();
	}
	delete element.mouseInButton;
	event.stopPropagation();
	event.preventDefault();
}

function snapbackMouseMove (event)
{
	// do nothing
	event.stopPropagation();
	event.preventDefault();
}

function snapbackMouseOver (event)
{
	var element = document.getElementById('SearchInputElementSnapbackIcon');
	
	if (!element.mouseInButton)
	{
		element.src = 'Images/Search_SnapBackPressed.png';
		element.mouseInButton = true;
	}
	event.stopPropagation();
	event.preventDefault();
}

function snapbackMouseOut (event)
{
	var element = document.getElementById('SearchInputElementSnapbackIcon');
	
	if (element.mouseInButton)
	{
		element.src = 'Images/Search_SnapBack.png';
		element.mouseInButton = false;
	}

	event.stopPropagation();
	event.preventDefault();
}

function keyPressed (event)
{
    var handled = false;
	
    if (event.metaKey) // command is down
    {
        var keyString = String.fromCharCode(event.charCode);
        if (keyString == '[')
        {
            Arrows.clickedLeftButton (event);
            handled = true;
        }
        else if (keyString == ']')
        {
            Arrows.clickedRightButton (event);
            handled = true;
        }
    }
    else if (event.keyCode == 13 || event.keyCode == 3) // return or enter
    {
		if (View.isCurrentViewDetailView())
		{
			if (event.shiftKey)
				Arrows.clickedLeftButton (event);
			else
				Arrows.clickedRightButton (event);
			handled = true;
		}
		else if (View.isCurrentViewNameListView())
		{
			if (!ExpandShrink._animation)
			{
				gABCurrentCardIndex = NameListView.active;
				View.setCurrent(CardDetailView);
				CardDetailView.refresh();
			}
			handled = true;
		}
    }
    else if (event.keyCode == 9) // tab key
	{
		var searchField = document.getElementById('SearchInputElement');
		var selectionStart = searchField.selectionStart,
			selectionEnd   = searchField.selectionEnd;
		searchField.focus();
		searchField.setSelectionRange(selectionStart, selectionEnd);
		
		handled = true;
	}
    else if (View.isCurrentViewNameListView())
	{
		if (!ExpandShrink._animation)
		{
	    	NameListView.keyPress(event);
	    }
	}
	
    if (handled)
    {
        event.stopPropagation();
        event.preventDefault();
    }
}
document.addEventListener("keypress", keyPressed, true);


/*======================================================================
 *	Debugging
 *======================================================================*/
function LOG__(arg)
{
	if (window.widget && gDisplayLog)
		alert(":"+arg);
	else if (gDisplayLog)
		debug(arg);
}

/*======================================================================
 *	End of the script
 *======================================================================*/
